//
// ViewController.swift
// Charts Sample
//
// These samples introduce how to create  UIChart object as  a specific Chart type
// that include Pie Graph, Bar Graph, Line Graph, Cylinder,  Area, and other chart
// types on your iOS app layout.
//
// The main idea of the samples is to provide you with a suggestion how to display
// tabular data values as a Chart.  Charts are often the best  way to  demonstrate
// trends in data  and make comparisons between different groups.  Different types
// of graphs are required to  most efficiently and effectively represent different
// types of data.
//
// Also the code example gives a hint how to deal  with source  data when you need
// manipulate and graphically display set of data records in the UIChart interface.
//
// Please read on-line documentation: http://rustemsoft.com/TouchControlsDoc/index.html
//
// Copyright (c) 2015 Rustemsoft LLC. All rights reserved.
//
// ======================================================================================
// Xcode 6.1.1 still has problems related to framework build and referencing.
// We have listed resolutions for couple of possible issues you may encounter:
//
// 1. When you run the Sample project for very first time the build may fail. To fix that
//   remove the reference to TouchControlsTrial.framework and re-reference to the library
//   as following:
// - In the Project Navigator, select the Sample project.
// - Select the target.
// - Select the "General" tab.
// - Open "Embedded Binaries" expander.
// - Click the + button.
// - Select the TouchControlsTrial.framework.
// - Window -> Organizer -> Delete Derived Data and then: Product -> Clean.
// =======================================================================================
//
// How to Manually Add a Swift Bridging Header?
//
// 1. Add a header file to your project, named:  MyProjectName-Bridging-Header.h.  This will be the
// single header file where you import any Objective-C code you want your Swift code to have access.
//
// 2. In your project build settings, find Swift Compiler – Code Generation, and next to Objective-C
// Bridging Header add the path to your bridging header file, from the project’s root folder.  So it
// could by MyProject/MyProject-Bridging-Header.h or simply MyProject-Bridging-Header.h  if the file
// lives in the project root folder.
//
// You only need one Bridging Header. Add your #import statements to this file,and your classes will
// now be available in your Swift code without any extra import statements.
//

import UIKit

// Device detection. Define size coefficient if it is iPhone or iPad
var COEFF_PAD: CGFloat { return ((UIDevice.currentDevice().userInterfaceIdiom == .Pad) ? 2.0 : 1.0) }

// Define public variables
var chrt = UIChart()
var btnForward = UIButton()
var btnBackward = UIButton()
var lblSampleTitle = UILabel()
var lblNegative = UILabel()
var swhOrientation = UISwitch()
var swhNegativeSeries = UISwitch()
var currentChartType: Int = 0
var dataItmsSeries0: NSMutableArray = NSMutableArray()
var dataItmsSeries1: NSMutableArray = NSMutableArray()
var dataItmsSeries2: NSMutableArray = NSMutableArray()

class ViewController: UIViewController, UIScrollViewDelegate, UIAlertViewDelegate, ChartDelegate {

    override func viewDidLoad() {
        super.viewDidLoad()
        
        // Set background image
        var img = UIImage(named: "background.png")
        var bgImg = UIImageView(image: img)
        bgImg.frame = CGRect(x: 0, y: 0, width: self.view.frame.width, height: self.view.frame.height)
        self.view.addSubview(bgImg)
        
        // Define data sources for chart items
        self.dataItemsSeries()

        // Draw Chart layout and plot
        self.defineChart()
        
        // Run Chart
        chrt.DataBind()
        
        // Add Orientation switch to define charts' Vertical/Horizontal orientation
        swhOrientation.frame = CGRect(x: COEFF_PAD * 50, y: chrt.frame.origin.y + chrt.frame.size.height + COEFF_PAD * 5, width: COEFF_PAD * 10, height: COEFF_PAD * 2)
        swhOrientation.transform = CGAffineTransformMakeScale(COEFF_PAD * 0.5, COEFF_PAD * 0.5)
        swhOrientation.addTarget(self, action: "swhChanged:", forControlEvents: .ValueChanged)
        swhOrientation.on = true
        swhOrientation.enabled = false
        self.view.addSubview(swhOrientation)
        let lblHorizontal = UILabel(frame: CGRect(x: swhOrientation.frame.origin.x - COEFF_PAD * 34, y: swhOrientation.frame.origin.y - 2, width: COEFF_PAD * 100, height: COEFF_PAD * 20))
        lblHorizontal.font = UIFont(name: "Georgia", size: COEFF_PAD * 7)
        lblHorizontal.text = "Horizontal"
        self.view.addSubview(lblHorizontal)
        let lblVertical = UILabel(frame: CGRect(x: swhOrientation.frame.origin.x + swhOrientation.frame.size.width + 2, y: swhOrientation.frame.origin.y - 2, width: COEFF_PAD * 100, height: COEFF_PAD * 20))
        lblVertical.font = UIFont(name: "Georgia", size: COEFF_PAD * 7)
        lblVertical.text = "Vertical"
        self.view.addSubview(lblVertical)
        
        // Add a switch to add Series with Negative values
        swhNegativeSeries.frame = CGRect(x: COEFF_PAD * 130, y: chrt.frame.origin.y + chrt.frame.size.height + COEFF_PAD * 5, width: COEFF_PAD * 10, height: COEFF_PAD * 2)
        swhNegativeSeries.transform = CGAffineTransformMakeScale(COEFF_PAD * 0.5, COEFF_PAD * 0.5)
        swhNegativeSeries.addTarget(self, action: "swhChanged:", forControlEvents: .ValueChanged)
        swhNegativeSeries.on = false
        self.view.addSubview(swhNegativeSeries)
        lblNegative.frame = CGRect(x: swhNegativeSeries.frame.origin.x + swhNegativeSeries.frame.size.width + 2, y: swhNegativeSeries.frame.origin.y - 2, width: COEFF_PAD * 100, height: COEFF_PAD * 20)
        lblNegative.font = UIFont(name: "Georgia", size: COEFF_PAD * 7)
        lblNegative.text = "non-3D pie"
        self.view.addSubview(lblNegative)
        
        // Add btnBackward button
        btnBackward.frame = CGRect(x: 0, y: chrt.frame.origin.y + chrt.frame.size.height / 2, width: COEFF_PAD * 15, height: COEFF_PAD * 15)
        btnBackward.setImage(UIImage(named: "back.png"), forState: .Normal)
        btnBackward.addTarget(self, action: "btnBackwardClicked:", forControlEvents: .TouchUpInside)
        btnBackward.alpha = 0.5
        self.view.addSubview(btnBackward)
        
        // Add btnForward button
        btnForward.frame = CGRect(x: chrt.frame.origin.x + chrt.frame.size.width - 5, y: chrt.frame.origin.y + chrt.frame.size.height / 2, width: COEFF_PAD * 15, height: COEFF_PAD * 15)
        btnForward.setImage(UIImage(named: "forw.png"), forState: .Normal)
        btnForward.addTarget(self, action: "btnForwardClicked:", forControlEvents: .TouchUpInside)
        self.view.addSubview(btnForward)
        
        // Define Sample Title label
        lblSampleTitle.frame = CGRect(x: COEFF_PAD * 10, y: COEFF_PAD * 15, width: COEFF_PAD * 220, height: COEFF_PAD * 20)
        lblSampleTitle.text = "Pie Chart"
        lblSampleTitle.textColor = UIColor.purpleColor()
        lblSampleTitle.font = UIFont(name: "HelveticaNeue-CondensedBold", size: COEFF_PAD * 16)
        self.view.addSubview(lblSampleTitle)
    }
    
    // Define Chart Layout
    func defineChart() {
        chrt.removeFromSuperview()
        
        // Create UIChart object
        chrt = UIChart(frame: CGRect(x: COEFF_PAD * 10, y: COEFF_PAD * 40, width: self.view.frame.size.width - COEFF_PAD * 40, height: self.view.frame.size.height - COEFF_PAD * 190))
        chrt.delegate = self // set UIChart's delegate to self (otherwise events will not work)
        
        // Add UIChart object into app's main view layout
        self.view.addSubview(chrt)
        // Define chart Type
        chrt.chartType = defineChartType()
        
        // Define chart's colors
        chrt.backgroundColor = UIColor(red: 0.01, green:0.01, blue:0.5, alpha:0.1)
        chrt.foregroundColor = UIColor(red: 0.9, green:0.9, blue:1, alpha:1)
        
        chrt.orientation = swhOrientation.on ? orientations.vertical : orientations.horizontal
        
        // Define chart plot area
        chrt.plotArea.backgroundColor = UIColor(red: 0.1, green:0.1, blue:0.8, alpha:0.1)
        
        // Draw all kinds of Chart Types
        if (chrt.chartType == chartTypes.pie) {
            lblSampleTitle.text = "Pie Chart"
            
            // Define chart's Layout style
            chrt.layoutStyle = layoutStyles.Gradient
            // Define chart's gradient
            chrt.gradientStyle = gradientStyles.Radial
            chrt.gradientRadius = 200
            chrt.gradientColorTop = UIColor(red: 0.1, green:0.8, blue:0.1, alpha:0.4)
            chrt.gradientColorTopExtra = UIColor.clearColor()
            chrt.gradientColorBottom = UIColor.clearColor()
            chrt.gradientColorBottomExtra = UIColor(red: 0.1, green:0.1, blue:0.8, alpha:0.4)
            chrt.gradientDirection = gradientDirections.Horizontal
            
            // Assign chart's Title
            chrt.chartTitle.text = "Who did eat more apples?"
            chrt.chartTitle.font = UIFont(name: "Georgia", size: COEFF_PAD * 16)
            chrt.chartTitle.backgroundColor = UIColor(red: 0.1, green:0.1, blue:0.8, alpha:0.1)
            chrt.chartTitle.textColor = UIColor.whiteColor()
            chrt.chartTitle.frame = CGRectMake(COEFF_PAD * 5, COEFF_PAD * 5, COEFF_PAD * 185, COEFF_PAD * 18)
            
            // Set X axis title
            chrt.xAxisTitleLabel.text = "Jonathan Apple Sauce"
            chrt.xAxisTitleLabel.textColor = UIColor.purpleColor()
            chrt.xAxisTitleLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 10)
            chrt.xAxisTitleHorizontalLocation = horizontalLocations.Mid
            
            // Set Y axis title
            chrt.yAxisTitleLabel.text = "McIntosh Apple Pie"
            chrt.yAxisTitleLabel.textColor = UIColor.purpleColor()
            chrt.yAxisTitleLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 10)
            chrt.yAxisTitleLabel.frame = CGRectMake(COEFF_PAD * -83, COEFF_PAD * 160, COEFF_PAD * 185, COEFF_PAD * 18)
            chrt.yAxisTitleVerticalLocation = verticalLocations.Mid
            
            // Draw Legend Zone
            chrt.legendZone.layer.borderColor = UIColor.yellowColor().CGColor
            chrt.legendZone.layer.borderWidth = COEFF_PAD * 0.5
            chrt.legendZone.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 8)
            chrt.legendZone.Title.text = "Persons"
            chrt.legendZone.Title.textColor = UIColor.yellowColor()
            chrt.legendZone.titleVerticalLocation = verticalLocations.Top
            chrt.legendZone.titleHorizontalLocation = horizontalLocations.Mid
            chrt.legendZoneHorizontalLocation = horizontalLocations.Left
            chrt.legendZoneVerticalLocation = verticalLocations.Bottom
            chrt.legendsInterpositions = orientations.vertical
            
            // Define chart series
            var srs0 = ChartSeries()
            var srs1 = ChartSeries()
            
            // 3D Pie or not
            srs0.pie3D = !swhNegativeSeries.on
            // Define line shadow
            srs0.shape.shadowColor = UIColor.blackColor().CGColor
            srs0.shape.shadowOpacity = 0.9
            srs0.shape.shadowOffset = CGSizeMake(COEFF_PAD * 10, COEFF_PAD * 10)
            srs0.shape.shadowRadius = 10
            srs0.shape.shadowPath = srs0.shape.path
            srs0.shape.lineWidth = COEFF_PAD * 0.3
            // Fill out chart items collection
            for var i = 0; i < dataItmsSeries0.count; i++ {
                var itm = dataItmsSeries0.objectAtIndex(i) as NSMutableArray
                let chIt = ChartItem()
                chIt.value = CGFloat((itm.objectAtIndex(1) as NSString).doubleValue)
                chIt.Title.text = itm.objectAtIndex(1) as NSString
                chIt.Title.textColor = UIColor.brownColor()
                chIt.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 8)
                chIt.layer.borderColor = UIColor.whiteColor().CGColor
                chIt.layer.borderWidth = COEFF_PAD * 0.3
                chIt.layoutStyle = layoutStyles.Flat
                chIt.titleVerticalLocation = verticalLocations.Bottom
                chIt.titleHorizontalLocation = horizontalLocations.Mid
                // Create legend bullet to add into the legend zone
                var lgd = legend()
                lgd.legendLabel.text = itm.objectAtIndex(0) as NSString
                lgd.legendLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 7)
                lgd.legendBulletType = legendBulletTypes.Rect
                // Specify legend bullet's color
                var colorHue: CGFloat = CGFloat(Double(i) / Double(dataItmsSeries0.count))
                lgd.legendBulletFillColor = UIColor(hue: colorHue, saturation:1, brightness:1, alpha:1)
                lgd.legendBulletWidth = COEFF_PAD * 10
                // Add legend bullet into legend zone
                chrt.legends.addObject(lgd)
                // Add chart item into series
                srs0.Items.addObject(chIt)
            }
            
            // 3D Pie or not
            srs1.pie3D = !swhNegativeSeries.on
            srs1.shape.lineWidth = COEFF_PAD * 0.3
            srs1.shape.strokeColor = UIColor.redColor().CGColor
            srs1.pieRadius = 65
            for var i = 0; i < dataItmsSeries1.count; i++ {
                var itm = dataItmsSeries1.objectAtIndex(i) as NSMutableArray
                let chIt = ChartItem()
                chIt.value = CGFloat((itm.objectAtIndex(1) as NSString).doubleValue)
                chIt.Title.text = itm.objectAtIndex(1) as NSString
                chIt.Title.textColor = UIColor.blackColor()
                chIt.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
                chIt.titleHorizontalLocation = horizontalLocations.Mid
                chIt.titleVerticalLocation = verticalLocations.Mid
                chIt.pointWidth = 0
                chIt.layoutStyle = layoutStyles.Flat
                // Add chart item into series
                srs1.Items.addObject(chIt)
            }
            
            // Add series into chart
            chrt.series.addObject(srs0)
            chrt.series.addObject(srs1)
        } else if (chrt.chartType == chartTypes.bar) {
            lblSampleTitle.text = "Bar Chart"
            
            // Define chart's Layout style
            chrt.layoutStyle = layoutStyles.Hatch
            chrt.hatchLineStepHeight = 8
            chrt.hatchLineWidth = COEFF_PAD * 0.1
            
            // Also you may define chart's layout as a background image
            //chrt.layoutStyle = layoutStyles.Image
            //chrt.backgroundImageName = "background.png"
            
            // Assign chart's Title
            chrt.chartTitle.text = "Apples have been eaten in 3 days"
            chrt.chartTitle.font = UIFont(name:"Georgia", size:COEFF_PAD * 16)
            chrt.chartTitle.backgroundColor = UIColor(red: 0.1, green:0.1, blue:0.8, alpha:0.1)
            chrt.chartTitle.textColor = UIColor.whiteColor()
            
            // Add marked region into chart plot area
            let markedRgn = markedRegion()
            markedRgn.yTop = 0.5
            markedRgn.yBottom = 0.375
            markedRgn.Title.text = "Marked Area"
            markedRgn.Title.textColor = UIColor.yellowColor()
            markedRgn.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 8)
            // Adjust marked region location
            markedRgn.titleVerticalLocation = verticalLocations.Top
            markedRgn.titleHorizontalLocation = horizontalLocations.Left
            chrt.markedRegions.addObject(markedRgn)
            
            // Set X axis title
            chrt.xAxisTitleLabel.text = "Persons"
            chrt.xAxisTitleLabel.textColor = UIColor.purpleColor()
            chrt.xAxisTitleLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 12)
            chrt.xAxisTitleHorizontalLocation = horizontalLocations.Right
            // X axis labels
            chrt.xAxisLinesHidden = false
            chrt.xAxisLabelsHidden = false
            chrt.xAxisLabelsFont = UIFont(name:"Georgia", size:COEFF_PAD * 6)
            chrt.xAxisLabelsTextColor = UIColor.yellowColor()
            // Y axis labels
            chrt.yAxisLabelsFont = UIFont(name:"Georgia", size:COEFF_PAD * 6)
            chrt.yAxisLabelsTextColor = UIColor.blueColor()
            
            // Draw Legend Zone
            chrt.legendZone.layer.borderColor = UIColor.yellowColor().CGColor
            chrt.legendZone.layer.borderWidth = COEFF_PAD * 0.5
            chrt.legendsInterpositions = orientations.horizontal
            chrt.legendZoneHorizontalLocation = horizontalLocations.Right
            chrt.legendZoneVerticalLocation = verticalLocations.Bottom
            
            // Create legend items to add into the legend zone
            var lgd = legend()
            lgd.legendLabel.text = "High value"
            lgd.legendLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 7)
            lgd.legendBulletType = legendBulletTypes.Image
            lgd.legendBulletImageName = "flwr_red.gif"
            lgd.legendBulletWidth = COEFF_PAD * 12
            chrt.legends.addObject(lgd)
            lgd = legend()
            lgd.legendLabel.text = "Normal value"
            lgd.legendLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 7)
            lgd.legendBulletType = legendBulletTypes.Image
            lgd.legendBulletImageName = "flwr_white.gif"
            lgd.legendBulletWidth = COEFF_PAD * 12
            chrt.legends.addObject(lgd)
            lgd = legend()
            lgd.legendLabel.text = "Low value"
            lgd.legendBulletType = legendBulletTypes.Image
            lgd.legendLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 7)
            lgd.legendBulletImageName = "flwr_blue.gif"
            lgd.legendBulletWidth = COEFF_PAD * 12
            chrt.legends.addObject(lgd)
            
            // Define chart series
            var srs0 = ChartSeries()
            var srs1 = ChartSeries()
            var srs2 = ChartSeries()
            
            // Fill out chart items of 1st series
            for var i = 0; i < dataItmsSeries0.count; i++ {
                var itm = dataItmsSeries0.objectAtIndex(i) as NSMutableArray
                let chIt = ChartItem()
                chIt.value = CGFloat((itm.objectAtIndex(1) as NSString).doubleValue)
                // Item Title text
                chIt.Title.text = itm.objectAtIndex(1) as NSString
                chIt.Title.textColor = UIColor.blackColor()
                chIt.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
                // Item Title location
                chIt.titleVerticalLocation = verticalLocations.Mid
                chIt.titleHorizontalLocation = horizontalLocations.Mid
                // Define item's border
                chIt.layer.borderColor = UIColor.whiteColor().CGColor
                chIt.layer.borderWidth = COEFF_PAD * 0.3
                chIt.layoutStyle = layoutStyles.Gradient
                chIt.gradientStyle = gradientStyles.Simple
                chIt.gradientDirection = gradientDirections.DiagonalLeft
                // X axis label text
                chIt.xAxisLabel.text = itm.objectAtIndex(0) as NSString
                
                // Apply colors depending on max/min/normal values
                if (chIt.value > markedRgn.yTop) { // high value
                    chIt.gradientColorTop = UIColor.redColor()
                    chIt.gradientColorTopExtra = UIColor(red: 1, green:0, blue:0, alpha:0.2)
                    chIt.gradientColorBottom = UIColor(red: 1, green:0, blue:0, alpha:0.2)
                } else if (chIt.value < markedRgn.yBottom) { // low value
                    chIt.gradientColorTop = UIColor.blueColor()
                    chIt.gradientColorTopExtra = UIColor(red: 0, green:0, blue:0.9, alpha:0.1)
                    chIt.gradientColorBottom = UIColor(red: 0, green:0, blue:0.9, alpha:0.1)
                } else { // normal range
                    chIt.gradientColorTop = UIColor.whiteColor()
                    chIt.gradientColorTopExtra = UIColor(red: 0.9, green:0.9, blue:0.9, alpha:0.05)
                    chIt.gradientColorBottom = UIColor(red: 0.9, green:0.9, blue:0.9, alpha:0.05)
                }
                // Add chart item into series
                srs0.Items.addObject(chIt)
            }
            
            // Fill out chart items of 2nd series
            for var i = 0; i < dataItmsSeries1.count; i++ {
                var itm = dataItmsSeries1.objectAtIndex(i) as NSMutableArray
                let chIt = ChartItem()
                chIt.value = CGFloat((itm.objectAtIndex(1) as NSString).doubleValue)
                chIt.Title.text = itm.objectAtIndex(1) as NSString
                chIt.Title.textColor = UIColor.whiteColor()
                chIt.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
                chIt.titleHorizontalLocation = horizontalLocations.Mid
                chIt.titleVerticalLocation = verticalLocations.Top
                chIt.layoutStyle = layoutStyles.Flat
                chIt.backgroundColor = UIColor.orangeColor()
                // Add chart item into series
                srs1.Items.addObject(chIt)
            }
            
            // Fill out chart items of 3rd series
            for var i = 0; i < dataItmsSeries2.count; i++ {
                var itm = dataItmsSeries2.objectAtIndex(i) as NSMutableArray
                let chIt = ChartItem()
                chIt.value = CGFloat((itm.objectAtIndex(1) as NSString).doubleValue) * (swhNegativeSeries.on ? -1 : 1)
                chIt.Title.text = NSString(format: "%.2f", chIt.value)
                chIt.Title.textColor = UIColor.yellowColor()
                chIt.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
                chIt.titleHorizontalLocation = horizontalLocations.Mid
                chIt.titleVerticalLocation = verticalLocations.Mid
                chIt.pointType = pointTypes.Rect
                // Define item's border
                chIt.layer.borderWidth = COEFF_PAD * 0.5
                // Define item's layout style
                chIt.layoutStyle = layoutStyles.Gradient
                chIt.gradientStyle = gradientStyles.Rainbow
                chIt.gradientDirection = gradientDirections.DiagonalRight
                // Also you may define item's layout style as an image
                //chIt.backgroundImageName = "background.png"
                // Add chart item into series
                srs2.Items.addObject(chIt)
            }
            // Add series into chart
            chrt.series.addObject(srs0)
            chrt.series.addObject(srs1)
            chrt.series.addObject(srs2)
            
            // Proportion width of bar/cylinder item (range: 0.0 - 1.0) Default value is 0.5
            chrt.itemBreadth = 0.6
        } else if (chrt.chartType == chartTypes.barStacked) {
            lblSampleTitle.text = "Stacked Bar Chart"
            
            // Define chart's Layout style
            chrt.layoutStyle = layoutStyles.Hatch
            chrt.hatchLineStepHeight = 8
            chrt.hatchLineWidth = COEFF_PAD * 0.1
            
            // Also you may define chart's layout as a background image
            //chrt.layoutStyle = layoutStyles.Image
            //chrt.backgroundImageName = "background.png"
            
            // Assign chart's Title
            chrt.chartTitle.text = "Apples have been eaten in 3 days"
            chrt.chartTitle.font = UIFont(name:"Georgia", size:COEFF_PAD * 16)
            chrt.chartTitle.backgroundColor = UIColor(red: 0.1, green:0.1, blue:0.8, alpha:0.1)
            chrt.chartTitle.textColor = UIColor.whiteColor()
            
            // Add marked region into chart plot area
            let markedRgn = markedRegion()
            markedRgn.yTop = 0.5
            markedRgn.yBottom = 0.375
            markedRgn.Title.text = "Marked Area"
            markedRgn.Title.textColor = UIColor.yellowColor()
            markedRgn.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 8)
            // Adjust marked region location
            markedRgn.titleVerticalLocation = verticalLocations.Top
            markedRgn.titleHorizontalLocation = horizontalLocations.Left
            chrt.markedRegions.addObject(markedRgn)
            
            // Set X axis title
            chrt.xAxisTitleLabel.text = "Persons"
            chrt.xAxisTitleLabel.textColor = UIColor.purpleColor()
            chrt.xAxisTitleLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 12)
            chrt.xAxisTitleHorizontalLocation = horizontalLocations.Right
            // X axis labels
            chrt.xAxisLinesHidden = false
            chrt.xAxisLabelsHidden = false
            chrt.xAxisLabelsFont = UIFont(name:"Georgia", size:COEFF_PAD * 6)
            chrt.xAxisLabelsTextColor = UIColor.yellowColor()
            // Y axis labels
            chrt.yAxisLabelsFont = UIFont(name:"Georgia", size:COEFF_PAD * 6)
            chrt.yAxisLabelsTextColor = UIColor.blueColor()
            
            // Draw Legend Zone
            chrt.legendZone.layer.borderColor = UIColor.yellowColor().CGColor
            chrt.legendZone.layer.borderWidth = COEFF_PAD * 0.5
            chrt.legendsInterpositions = orientations.horizontal
            chrt.legendZoneHorizontalLocation = horizontalLocations.Right
            chrt.legendZoneVerticalLocation = verticalLocations.Bottom
            
            // Create legend items to add into the legend zone
           var lgd = legend()
            lgd.legendLabel.text = "High value"
            lgd.legendLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 7)
            lgd.legendBulletType = legendBulletTypes.Image
            lgd.legendBulletImageName = "flwr_red.gif"
            lgd.legendBulletWidth = COEFF_PAD * 12
            chrt.legends.addObject(lgd)
            lgd = legend()
            lgd.legendLabel.text = "Normal value"
            lgd.legendLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 7)
            lgd.legendBulletType = legendBulletTypes.Image
            lgd.legendBulletImageName = "flwr_white.gif"
            lgd.legendBulletWidth = COEFF_PAD * 12
            chrt.legends.addObject(lgd)
            lgd = legend()
            lgd.legendLabel.text = "Low value"
            lgd.legendBulletType = legendBulletTypes.Image
            lgd.legendLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 7)
            lgd.legendBulletImageName = "flwr_blue.gif"
            lgd.legendBulletWidth = COEFF_PAD * 12
            chrt.legends.addObject(lgd)
            
            // Define chart series
            let srs0 = ChartSeries()
            let srs1 = ChartSeries()
            let srs2 = ChartSeries()
            
            // Fill out chart items of 1st series
            for var i = 0; i < dataItmsSeries0.count; i++ {
                var itm = dataItmsSeries0.objectAtIndex(i) as NSMutableArray
                let chIt = ChartItem()
                chIt.value = CGFloat((itm.objectAtIndex(1) as NSString).doubleValue)
                // Item Title text
                chIt.Title.text = itm.objectAtIndex(1) as NSString
                chIt.Title.textColor = UIColor.blackColor()
                chIt.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
                // Item Title location
                chIt.titleVerticalLocation = verticalLocations.Mid
                chIt.titleHorizontalLocation = horizontalLocations.Mid
                // Define item's border
                chIt.layer.borderColor = UIColor.whiteColor().CGColor
                chIt.layer.borderWidth = COEFF_PAD * 0.3
                chIt.layoutStyle = layoutStyles.Gradient
                chIt.gradientStyle = gradientStyles.Simple
                chIt.gradientDirection = gradientDirections.DiagonalLeft
                // X axis label text
                chIt.xAxisLabel.text = itm.objectAtIndex(0) as NSString as NSString
                
                // Apply colors depending on max/min/normal values
                if (chIt.value > markedRgn.yTop) { // high value
                    chIt.gradientColorTop = UIColor.redColor()
                    chIt.gradientColorTopExtra = UIColor(red: 1, green:0, blue:0, alpha:0.2)
                    chIt.gradientColorBottom = UIColor(red: 1, green:0, blue:0, alpha:0.2)
                } else if (chIt.value < markedRgn.yBottom) { // low value
                    chIt.gradientColorTop = UIColor.blueColor()
                    chIt.gradientColorTopExtra = UIColor(red: 0, green:0, blue:0.9, alpha:0.1)
                    chIt.gradientColorBottom = UIColor(red: 0, green:0, blue:0.9, alpha:0.1)
                } else { // normal range
                    chIt.gradientColorTop = UIColor.whiteColor()
                    chIt.gradientColorTopExtra = UIColor(red: 0.9, green:0.9, blue:0.9, alpha:0.05)
                    chIt.gradientColorBottom = UIColor(red: 0.9, green:0.9, blue:0.9, alpha:0.05)
                }
                // Add chart item into series
                srs0.Items.addObject(chIt)
            }
            
            // Fill out chart items of 2nd series
            for var i = 0; i < dataItmsSeries1.count; i++ {
                var itm = dataItmsSeries1.objectAtIndex(i) as NSMutableArray
                let chIt = ChartItem()
                chIt.value = CGFloat((itm.objectAtIndex(1) as NSString).doubleValue)
                chIt.Title.text = itm.objectAtIndex(1) as NSString
                chIt.Title.textColor = UIColor.darkGrayColor()
                chIt.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
                chIt.titleHorizontalLocation = horizontalLocations.Mid
                chIt.titleVerticalLocation = verticalLocations.Mid
                chIt.layoutStyle = layoutStyles.Flat
                chIt.backgroundColor = UIColor.cyanColor()
                // Add chart item into series
                srs1.Items.addObject(chIt)
            }
            
            // Fill out chart items of 3rd series
            for var i = 0; i < dataItmsSeries2.count; i++ {
                var itm = dataItmsSeries2.objectAtIndex(i) as NSMutableArray
                let chIt = ChartItem()
                chIt.value = CGFloat((itm.objectAtIndex(1) as NSString).doubleValue) * (swhNegativeSeries.on ? -1 : 1)
                chIt.Title.text = NSString(format: "%.2f", chIt.value)
                chIt.Title.textColor = UIColor.yellowColor()
                chIt.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
                chIt.titleHorizontalLocation = horizontalLocations.Mid
                chIt.titleVerticalLocation = verticalLocations.Mid
                chIt.pointType = pointTypes.Rect
                // Define item's border
                chIt.layer.borderWidth = COEFF_PAD * 0.5
                // Define item's layout style
                chIt.layoutStyle = layoutStyles.Gradient
                chIt.gradientStyle = gradientStyles.Rainbow
                chIt.gradientDirection = gradientDirections.DiagonalRight
                // Also you may define item's layout style as an image
                //chIt.backgroundImageName = "background.png"
                // Add chart item into series
                srs2.Items.addObject(chIt)
            }
            // Add series into chart
            chrt.series.addObject(srs0)
            chrt.series.addObject(srs1)
            chrt.series.addObject(srs2)
            
            // Proportion width of bar/cylinder item (range: 0.0 - 1.0) Default value is 0.5
            chrt.itemBreadth = 0.6
        } else if (chrt.chartType == chartTypes.line) {
            lblSampleTitle.text = "Line Chart"
            
            // Define chart's Layout style
            chrt.layoutStyle = layoutStyles.Gradient
            // Define chart's gradient
            chrt.gradientStyle = gradientStyles.Radial
            chrt.gradientRadius = 200
            chrt.gradientColorTop = UIColor(red: 0.1, green:0.8, blue:0.1, alpha:0.4)
            chrt.gradientColorTopExtra = UIColor.clearColor()
            chrt.gradientColorBottom = UIColor.clearColor()
            chrt.gradientColorBottomExtra = UIColor(red: 0.1, green:0.1, blue:0.8, alpha:0.4)
            chrt.gradientDirection = gradientDirections.Horizontal
            
            // Assign chart's Title
            chrt.chartTitle.text = "Prevalence of 3 types of apples across Continents"
            chrt.chartTitle.font = UIFont(name:"Georgia", size:COEFF_PAD * 13)
            chrt.chartTitle.backgroundColor = UIColor(red: 0.1, green:0.1, blue:0.8, alpha:0.1)
            chrt.chartTitle.textColor = UIColor.whiteColor()
            chrt.chartTitle.frame = CGRectMake(COEFF_PAD * 5, COEFF_PAD * 5, COEFF_PAD * 310, COEFF_PAD * 18)
            
            // Add marked region into chart plot area
            let markedRgn = markedRegion()
            markedRgn.yTop = 0.5
            markedRgn.yBottom = 0.3
            markedRgn.Title.text = "Marked Area"
            markedRgn.Title.textColor = UIColor.yellowColor()
            markedRgn.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 8)
            // Adjust marked region location
            markedRgn.titleVerticalLocation = verticalLocations.Top
            markedRgn.titleHorizontalLocation = horizontalLocations.Right
            chrt.markedRegions.addObject(markedRgn)
            
            // Set X axis title
            chrt.xAxisTitleLabel.text = "Continents"
            chrt.xAxisTitleLabel.textColor = UIColor.purpleColor()
            chrt.xAxisTitleLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 12)
            chrt.xAxisTitleHorizontalLocation = horizontalLocations.Right
            // X axis labels
            chrt.xAxisLinesHidden = false
            chrt.xAxisLabelsHidden = false
            chrt.xAxisLabelsFont = UIFont(name:"Georgia", size:COEFF_PAD * 6)
            chrt.xAxisLabelsTextColor = UIColor.yellowColor()
            // Y axis labels
            chrt.yAxisLabelsFont = UIFont(name:"Georgia", size:COEFF_PAD * 6)
            chrt.yAxisLabelsTextColor = UIColor.blueColor()
            
            // Draw Legend Zone
            chrt.legendZone.layer.borderColor = UIColor.yellowColor().CGColor
            chrt.legendZone.layer.borderWidth = COEFF_PAD * 0.5
            chrt.legendsInterpositions = orientations.horizontal
            chrt.legendZoneHorizontalLocation = horizontalLocations.Mid
            chrt.legendZoneVerticalLocation = verticalLocations.Bottom
            
            // Create legend items to add into the legend zone
           var lgd = legend()
            lgd.legendLabel.text = "Rejuvenating Apple"
            lgd.legendLabel.textColor = UIColor.whiteColor()
            lgd.legendLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 7)
            lgd.legendBulletType = legendBulletTypes.Triangle
            lgd.legendBulletWidth = COEFF_PAD * 12
            lgd.legendBulletFillColor = UIColor.whiteColor()
            lgd.legendBulletStrokeLineColor = UIColor.redColor()
            lgd.legendBulletStrokeLineWidth = COEFF_PAD * 2
            lgd.legendMargin = COEFF_PAD * 3
            chrt.legends.addObject(lgd)
            lgd = legend()
            lgd.legendLabel.text = "Apple of Discord"
            lgd.legendLabel.textColor = UIColor.whiteColor()
            lgd.legendLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 7)
            lgd.legendBulletType = legendBulletTypes.Dot
            lgd.legendBulletFillColor = UIColor.redColor()
            lgd.legendBulletWidth = COEFF_PAD * 12
            lgd.legendMargin = COEFF_PAD * 3
            chrt.legends.addObject(lgd)
            lgd = legend()
            lgd.legendLabel.text = "Hesperides Apple"
            lgd.legendLabel.textColor = UIColor.whiteColor()
            lgd.legendLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 7)
            lgd.legendBulletType = legendBulletTypes.Rect
            lgd.legendBulletFillColor = UIColor.blueColor()
            lgd.legendBulletWidth = COEFF_PAD * 12
            lgd.legendMargin = COEFF_PAD * 3
            chrt.legends.addObject(lgd)
            
            // Define chart series
            let srs0 = ChartSeries()
            let srs1 = ChartSeries()
            let srs2 = ChartSeries()
            
            // Fill out chart items of 1st series
            // Define the series' line shape
            srs0.shape.lineWidth = COEFF_PAD * 0.3
            for var i = 0; i < dataItmsSeries1.count; i++ {
                var itm = dataItmsSeries1.objectAtIndex(i) as NSMutableArray
                let chIt = ChartItem()
                chIt.value = CGFloat((itm.objectAtIndex(1) as NSString).doubleValue)
                chIt.Title.text = itm.objectAtIndex(1) as NSString
                chIt.Title.textColor = UIColor.whiteColor()
                chIt.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
                chIt.layer.borderColor = UIColor.whiteColor().CGColor
                chIt.layer.borderWidth = COEFF_PAD * 0.3
                chIt.titleVerticalLocation = verticalLocations.Bottom
                chIt.titleHorizontalLocation = horizontalLocations.Mid
                chIt.pointType = pointTypes.Triangle
                chIt.pointWidth = COEFF_PAD * 5
                chIt.pointFillColor = UIColor.whiteColor()
                chIt.pointStrokeLineWidth = COEFF_PAD * 1
                chIt.pointStrokeLineColor = UIColor.redColor()
                chIt.layoutStyle = layoutStyles.Flat
                // X axis label text
                chIt.xAxisLabel.text = itm.objectAtIndex(0) as NSString
                // Add chart item into series
                srs0.Items.addObject(chIt)
            }
            
            // Fill out chart items of 2nd series
            // Define the series' line shape
            srs1.shape.lineWidth = COEFF_PAD * 3
            srs1.shape.strokeColor = UIColor.redColor().CGColor
            // Define line shadow
            srs1.shape.shadowColor = UIColor.blackColor().CGColor
            srs1.shape.shadowOpacity = 0.9
            srs1.shape.shadowOffset = CGSizeMake(COEFF_PAD * 10, COEFF_PAD * 10)
            srs1.shape.shadowRadius = 10
            srs1.shape.shadowPath = srs0.shape.path
            for var i = 0; i < dataItmsSeries0.count; i++ {
                var itm = dataItmsSeries0.objectAtIndex(i) as NSMutableArray
                let chIt = ChartItem()
                chIt.value = CGFloat((itm.objectAtIndex(1) as NSString).doubleValue)
                chIt.Title.text = itm.objectAtIndex(1) as NSString
                chIt.Title.textColor = UIColor.blackColor()
                chIt.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
                chIt.titleVerticalLocation = verticalLocations.Top
                chIt.titleHorizontalLocation = horizontalLocations.Mid
                chIt.pointWidth = 0
                chIt.pointType = pointTypes.Dot
                chIt.pointWidth = COEFF_PAD * 7
                chIt.layoutStyle = layoutStyles.Gradient
                chIt.gradientStyle = gradientStyles.Radial
                chIt.gradientRadius = COEFF_PAD * 4
                chIt.gradientColorTop = UIColor.redColor()
                chIt.gradientColorBottomExtra = UIColor.redColor()
                chIt.layer.borderWidth = COEFF_PAD * 1
                chIt.backgroundColor = UIColor.orangeColor()
                // Add chart item into series
                srs1.Items.addObject(chIt)
            }
            
            // Fill out chart items of 3rd series
            // Define the series' line shape
            srs2.shape.lineWidth = COEFF_PAD * 3
            srs2.shape.strokeColor = UIColor.blueColor().CGColor
            // Define line shadow
            srs2.shape.shadowColor = UIColor.blackColor().CGColor
            srs2.shape.shadowOpacity = 0.9
            srs2.shape.shadowOffset = CGSizeMake(COEFF_PAD * 10, COEFF_PAD * 10)
            srs2.shape.shadowRadius = 10
            srs2.shape.shadowPath = srs0.shape.path
            for var i = 0; i < dataItmsSeries2.count; i++ {
                var itm = dataItmsSeries2.objectAtIndex(i) as NSMutableArray
                let chIt = ChartItem()
                chIt.value = CGFloat((itm.objectAtIndex(1) as NSString).doubleValue) * (swhNegativeSeries.on ? -1 : 1)
                chIt.Title.text = NSString(format: "%.2f", chIt.value)
                chIt.Title.textColor = UIColor.blueColor()
                chIt.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
                chIt.titleVerticalLocation = verticalLocations.Top
                chIt.titleHorizontalLocation = horizontalLocations.Mid
                chIt.pointType = pointTypes.Rect
                chIt.pointWidth = COEFF_PAD * 7
                chIt.layer.borderWidth = COEFF_PAD * 1
                chIt.layoutStyle = layoutStyles.Gradient
                chIt.gradientStyle = gradientStyles.Radial
                chIt.gradientRadius = COEFF_PAD * 3
                chIt.backgroundColor = UIColor.brownColor()
                // Add chart item into series
                srs2.Items.addObject(chIt)
            }
            // Add series into chart
            chrt.series.addObject(srs0)
            chrt.series.addObject(srs1)
            chrt.series.addObject(srs2)
        } else if (chrt.chartType == chartTypes.area) {
            lblSampleTitle.text = "Area Chart"
            
            // Define chart's Layout style
            chrt.layoutStyle = layoutStyles.Gradient
            // Define chart's gradient
            chrt.gradientStyle = gradientStyles.Radial
            chrt.gradientRadius = 200
            chrt.gradientColorTop = UIColor(red: 0.1, green:0.8, blue:0.1, alpha:0.4)
            chrt.gradientColorTopExtra = UIColor.clearColor()
            chrt.gradientColorBottom = UIColor.clearColor()
            chrt.gradientColorBottomExtra = UIColor(red: 0.1, green:0.1, blue:0.8, alpha:0.4)
            chrt.gradientDirection = gradientDirections.Horizontal
            
            // Assign chart's Title
            chrt.chartTitle.text = "Apple Comparison Chart"
            chrt.chartTitle.font = UIFont(name:"Georgia", size:COEFF_PAD * 16)
            chrt.chartTitle.backgroundColor = UIColor(red: 0.1, green:0.1, blue:0.8, alpha:0.1)
            chrt.chartTitle.textColor = UIColor.whiteColor()
            chrt.chartTitle.frame = CGRectMake(COEFF_PAD * 5, COEFF_PAD * 5, COEFF_PAD * 310, COEFF_PAD * 18)
            
            // Set X axis title
            chrt.xAxisTitleLabel.text = "Recommended Usage"
            chrt.xAxisTitleLabel.textColor = UIColor.lightTextColor()
            chrt.xAxisTitleLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 12)
            chrt.xAxisTitleHorizontalLocation = horizontalLocations.Right
            chrt.xAxisTitleYshift = 11
            // X axis labels
            chrt.xAxisLinesHidden = false
            chrt.xAxisLabelsHidden = false
            chrt.xAxisLabelsFont = UIFont(name:"Georgia", size:COEFF_PAD * 6)
            chrt.xAxisLabelsTextColor = UIColor.yellowColor()
            // Y axis labels
            chrt.yAxisLabelsFont = UIFont(name:"Georgia", size:COEFF_PAD * 6)
            chrt.yAxisLabelsTextColor = UIColor.blueColor()
            chrt.xAxisLabelsShowDefaultOrderNumbers = false
            
            // Draw Legend Zone
            chrt.legendZone.layer.borderColor = UIColor.brownColor().CGColor
            chrt.legendZone.layer.borderWidth = COEFF_PAD * 0.3
            chrt.legendsInterpositions = orientations.horizontal
            chrt.legendZoneHorizontalLocation = horizontalLocations.Left
            chrt.legendZoneVerticalLocation = verticalLocations.Bottom
            
            // Create legend items to add into the legend zone
           var lgd = legend()
            lgd.legendLabel.text = "McIntosh"
            lgd.legendLabel.textColor = UIColor.whiteColor()
            lgd.legendLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 7)
            lgd.legendBulletType = legendBulletTypes.Rect
            lgd.legendBulletFillColor = UIColor.whiteColor()
            lgd.legendBulletWidth = COEFF_PAD * 12
            lgd.legendMargin = COEFF_PAD * 3
            chrt.legends.addObject(lgd)
            lgd = legend()
            lgd.legendLabel.text = "Braeburn"
            lgd.legendLabel.textColor = UIColor.whiteColor()
            lgd.legendLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 7)
            lgd.legendBulletType = legendBulletTypes.Rect
            lgd.legendBulletFillColor = UIColor.redColor()
            lgd.legendBulletWidth = COEFF_PAD * 12
            lgd.legendMargin = COEFF_PAD * 3
            chrt.legends.addObject(lgd)
            lgd = legend()
            lgd.legendLabel.text = "Stayman"
            lgd.legendLabel.textColor = UIColor.whiteColor()
            lgd.legendLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 7)
            lgd.legendBulletType = legendBulletTypes.Rect
            lgd.legendBulletFillColor = UIColor.blueColor()
            lgd.legendBulletWidth = COEFF_PAD * 12
            lgd.legendMargin = COEFF_PAD * 3
            chrt.legends.addObject(lgd)
            
            // Define chart series
            let srs0 = ChartSeries()
            let srs1 = ChartSeries()
            let srs2 = ChartSeries()
            
            // Fill out chart items of 1st series
            for var i = 0; i < dataItmsSeries1.count; i++ {
                var itm = dataItmsSeries1.objectAtIndex(i) as NSMutableArray
                let chIt = ChartItem()
                chIt.value = CGFloat((itm.objectAtIndex(1) as NSString).doubleValue)
                chIt.Title.text = itm.objectAtIndex(1) as NSString
                chIt.Title.textColor = UIColor.whiteColor()
                chIt.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
                chIt.layer.borderColor = UIColor.whiteColor().CGColor
                chIt.layer.borderWidth = COEFF_PAD * 0.3
                chIt.titleVerticalLocation = verticalLocations.Top
                chIt.titleHorizontalLocation = horizontalLocations.Mid
                chIt.layoutStyle = layoutStyles.Gradient
                chIt.gradientStyle = gradientStyles.Simple
                chIt.gradientColorTop = UIColor.whiteColor()
                chIt.gradientColorTopExtra = UIColor(red: 0.9, green:0.9, blue:0.9, alpha:0.2)
                chIt.gradientColorBottom = UIColor(red: 0.9, green:0.9, blue:0.9, alpha:0.2)
                // Add chart item into series
                srs0.Items.addObject(chIt)
            }
            
            // Fill out chart items of 2nd series
            // Define the series' line shape
            srs1.shape.lineWidth = COEFF_PAD * 3
            srs1.shape.strokeColor = UIColor.redColor().CGColor
            // Define line shadow
            srs1.shape.shadowColor = UIColor.blackColor().CGColor
            srs1.shape.shadowOpacity = 0.9
            srs1.shape.shadowOffset = CGSizeMake(COEFF_PAD * 10, COEFF_PAD * 10)
            srs1.shape.shadowRadius = 10
            srs1.shape.shadowPath = srs0.shape.path
            for var i = 0; i < dataItmsSeries0.count; i++ {
                var itm = dataItmsSeries0.objectAtIndex(i) as NSMutableArray
                let chIt = ChartItem()
                chIt.value = CGFloat((itm.objectAtIndex(1) as NSString).doubleValue)
                chIt.Title.text = itm.objectAtIndex(1) as NSString
                chIt.Title.textColor = UIColor.blackColor()
                chIt.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
                chIt.titleVerticalLocation = verticalLocations.Top
                chIt.titleHorizontalLocation = horizontalLocations.Mid
                chIt.pointWidth = 0
                chIt.pointType = pointTypes.Dot
                chIt.pointWidth = COEFF_PAD * 7
                chIt.layoutStyle = layoutStyles.Gradient
                chIt.gradientStyle = gradientStyles.Radial
                chIt.gradientRadius = COEFF_PAD * 4
                chIt.gradientColorTop = UIColor.redColor()
                chIt.gradientColorBottomExtra = UIColor.redColor()
                chIt.layer.borderWidth = COEFF_PAD * 1
                chIt.backgroundColor = UIColor.orangeColor()
                // Add chart item into series
                srs1.Items.addObject(chIt)
            }
            
            // Fill out chart items of 3rd series
            // Define the series' line shape
            srs2.shape.lineWidth = COEFF_PAD * 3
            srs2.shape.strokeColor = UIColor.blueColor().CGColor
            // Define line shadow
            srs2.shape.shadowColor = UIColor.blackColor().CGColor
            srs2.shape.shadowOpacity = 0.9
            srs2.shape.shadowOffset = CGSizeMake(COEFF_PAD * 10, COEFF_PAD * 10)
            srs2.shape.shadowRadius = 10
            srs2.shape.shadowPath = srs0.shape.path
            for var i = 0; i < dataItmsSeries2.count; i++ {
                var itm = dataItmsSeries2.objectAtIndex(i) as NSMutableArray
                let chIt = ChartItem()
                chIt.value = CGFloat((itm.objectAtIndex(1) as NSString).doubleValue) * (swhNegativeSeries.on ? -1 : 1)
                chIt.Title.text = NSString(format: "%.2f", chIt.value)
                chIt.Title.textColor = UIColor.blueColor()
                chIt.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
                chIt.titleVerticalLocation = verticalLocations.Top
                chIt.titleHorizontalLocation = horizontalLocations.Mid
                chIt.pointType = pointTypes.Rect
                chIt.pointWidth = COEFF_PAD * 7
                chIt.layer.borderWidth = COEFF_PAD * 1
                chIt.layoutStyle = layoutStyles.Gradient
                chIt.gradientStyle = gradientStyles.Radial
                chIt.gradientRadius = COEFF_PAD * 3
                chIt.backgroundColor = UIColor.brownColor()
                // X axis label text
                chIt.xAxisLabel.text = itm.objectAtIndex(0) as NSString
                // Add chart item into series
                srs2.Items.addObject(chIt)
            }
            // Add series into chart
            chrt.series.addObject(srs0)
            chrt.series.addObject(srs1)
            chrt.series.addObject(srs2)
        } else if (chrt.chartType == chartTypes.areaStacked) {
            lblSampleTitle.text = "Stacked Area Chart"
            
            // Define chart's Layout style
            chrt.layoutStyle = layoutStyles.Gradient
            // Define chart's gradient
            chrt.gradientStyle = gradientStyles.Radial
            chrt.gradientRadius = 200
            chrt.gradientColorTop = UIColor(red: 0.1, green:0.8, blue:0.1, alpha:0.4)
            chrt.gradientColorTopExtra = UIColor.clearColor()
            chrt.gradientColorBottom = UIColor.clearColor()
            chrt.gradientColorBottomExtra = UIColor(red: 0.1, green:0.1, blue:0.8, alpha:0.4)
            chrt.gradientDirection = gradientDirections.Horizontal
            
            // Assign chart's Title
            chrt.chartTitle.text = "Apple Comparison Chart"
            chrt.chartTitle.font = UIFont(name:"Georgia", size:COEFF_PAD * 16)
            chrt.chartTitle.backgroundColor = UIColor(red: 0.1, green:0.1, blue:0.8, alpha:0.1)
            chrt.chartTitle.textColor = UIColor.whiteColor()
            chrt.chartTitle.frame = CGRectMake(COEFF_PAD * 5, COEFF_PAD * 5, COEFF_PAD * 310, COEFF_PAD * 18)
            
            // Set X axis title
            chrt.xAxisTitleLabel.text = "Recommended Usage"
            chrt.xAxisTitleLabel.textColor = UIColor.lightTextColor()
            chrt.xAxisTitleLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 12)
            chrt.xAxisTitleHorizontalLocation = horizontalLocations.Right
            chrt.xAxisTitleYshift = 11
            // X axis labels
            chrt.xAxisLinesHidden = false
            chrt.xAxisLabelsHidden = false
            chrt.xAxisLabelsFont = UIFont(name:"Georgia", size:COEFF_PAD * 6)
            chrt.xAxisLabelsTextColor = UIColor.yellowColor()
            // Y axis labels
            chrt.yAxisLabelsFont = UIFont(name:"Georgia", size:COEFF_PAD * 6)
            chrt.yAxisLabelsTextColor = UIColor.blueColor()
            chrt.xAxisLabelsShowDefaultOrderNumbers = false
            
            // Draw Legend Zone
            chrt.legendZone.layer.borderColor = UIColor.brownColor().CGColor
            chrt.legendZone.layer.borderWidth = COEFF_PAD * 0.3
            chrt.legendsInterpositions = orientations.horizontal
            chrt.legendZoneHorizontalLocation = horizontalLocations.Left
            chrt.legendZoneVerticalLocation = verticalLocations.Bottom
            
            // Create legend items to add into the legend zone
           var lgd = legend()
            lgd.legendLabel.text = "McIntosh"
            lgd.legendLabel.textColor = UIColor.whiteColor()
            lgd.legendLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 7)
            lgd.legendBulletType = legendBulletTypes.Rect
            lgd.legendBulletFillColor = UIColor.whiteColor()
            lgd.legendBulletWidth = COEFF_PAD * 12
            lgd.legendMargin = COEFF_PAD * 3
            chrt.legends.addObject(lgd)
            lgd = legend()
            lgd.legendLabel.text = "Braeburn"
            lgd.legendLabel.textColor = UIColor.whiteColor()
            lgd.legendLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 7)
            lgd.legendBulletType = legendBulletTypes.Rect
            lgd.legendBulletFillColor = UIColor.redColor()
            lgd.legendBulletWidth = COEFF_PAD * 12
            lgd.legendMargin = COEFF_PAD * 3
            chrt.legends.addObject(lgd)
            lgd = legend()
            lgd.legendLabel.text = "Stayman"
            lgd.legendLabel.textColor = UIColor.whiteColor()
            lgd.legendLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 7)
            lgd.legendBulletType = legendBulletTypes.Rect
            lgd.legendBulletFillColor = UIColor.blueColor()
            lgd.legendBulletWidth = COEFF_PAD * 12
            lgd.legendMargin = COEFF_PAD * 3
            chrt.legends.addObject(lgd)
            
            // Define chart series
            let srs0 = ChartSeries()
            let srs1 = ChartSeries()
            let srs2 = ChartSeries()
            
            // Fill out chart items of 1st series
            // Define the series' line shape
            srs0.shape.lineWidth = COEFF_PAD * 0.3
            for var i = 0; i < dataItmsSeries2.count; i++ {
                var itm = dataItmsSeries2.objectAtIndex(i) as NSMutableArray
                let chIt = ChartItem()
                chIt.value = CGFloat((itm.objectAtIndex(1) as NSString).doubleValue)
                chIt.Title.text = itm.objectAtIndex(1) as NSString
                chIt.Title.textColor = UIColor.whiteColor()
                chIt.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
                chIt.layer.borderColor = UIColor.whiteColor().CGColor
                chIt.layer.borderWidth = COEFF_PAD * 0.3
                chIt.titleVerticalLocation = verticalLocations.Bottom
                chIt.titleHorizontalLocation = horizontalLocations.Mid
                chIt.layoutStyle = layoutStyles.Gradient
                chIt.gradientStyle = gradientStyles.Simple
                chIt.gradientColorTop = UIColor.whiteColor()
                chIt.gradientColorTopExtra = UIColor(red: 0.9, green:0.9, blue:0.9, alpha:0.9)
                chIt.gradientColorBottom = UIColor(red: 0.9, green:0.9, blue:0.9, alpha:0.2)
                // X axis label text
                chIt.xAxisLabel.text = itm.objectAtIndex(0) as NSString
                // Add chart item into series
                srs0.Items.addObject(chIt)
            }
            
            // Fill out chart items of 2nd series
            // Define the series' line shape
            srs1.shape.lineWidth = COEFF_PAD * 3
            srs1.shape.strokeColor = UIColor.redColor().CGColor
            // Define line shadow
            srs1.shape.shadowColor = UIColor.blackColor().CGColor
            srs1.shape.shadowOpacity = 0.9
            srs1.shape.shadowOffset = CGSizeMake(COEFF_PAD * 10, COEFF_PAD * 10)
            srs1.shape.shadowRadius = 10
            srs1.shape.shadowPath = srs0.shape.path
            for var i = 0; i < dataItmsSeries0.count; i++ {
                var itm = dataItmsSeries0.objectAtIndex(i) as NSMutableArray
                let chIt = ChartItem()
                chIt.value = CGFloat((itm.objectAtIndex(1) as NSString).doubleValue)
                chIt.Title.text = itm.objectAtIndex(1) as NSString
                chIt.Title.textColor = UIColor.blackColor()
                chIt.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
                chIt.titleVerticalLocation = verticalLocations.Top
                chIt.titleHorizontalLocation = horizontalLocations.Mid
                chIt.pointWidth = 0
                chIt.pointType = pointTypes.Dot
                chIt.pointWidth = COEFF_PAD * 7
                chIt.layoutStyle = layoutStyles.Gradient
                chIt.gradientStyle = gradientStyles.Radial
                chIt.gradientRadius = COEFF_PAD * 4
                chIt.gradientColorTop = UIColor.redColor()
                chIt.gradientColorBottomExtra = UIColor.redColor()
                chIt.layer.borderWidth = COEFF_PAD * 1
                chIt.backgroundColor = UIColor.orangeColor()
                // Add chart item into series
                srs1.Items.addObject(chIt)
            }
            
            // Fill out chart items of 3rd series
            // Define the series' line shape
            srs2.shape.lineWidth = COEFF_PAD * 3
            srs2.shape.strokeColor = UIColor.blueColor().CGColor
            // Define line shadow
            srs2.shape.shadowColor = UIColor.blackColor().CGColor
            srs2.shape.shadowOpacity = 0.9
            srs2.shape.shadowOffset = CGSizeMake(COEFF_PAD * 10, COEFF_PAD * 10)
            srs2.shape.shadowRadius = 10
            srs2.shape.shadowPath = srs0.shape.path
            for var i = 0; i < dataItmsSeries1.count; i++ {
                var itm = dataItmsSeries1.objectAtIndex(i) as NSMutableArray
                let chIt = ChartItem()
                chIt.value = CGFloat((itm.objectAtIndex(1) as NSString).doubleValue) * (swhNegativeSeries.on ? -1 : 1)
                chIt.Title.text = NSString(format: "%.2f", chIt.value)
                chIt.Title.textColor = UIColor.blueColor()
                chIt.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
                chIt.titleVerticalLocation = verticalLocations.Top
                chIt.titleHorizontalLocation = horizontalLocations.Mid
                chIt.pointType = pointTypes.Rect
                chIt.pointWidth = COEFF_PAD * 7
                chIt.layer.borderWidth = COEFF_PAD * 1
                chIt.layoutStyle = layoutStyles.Gradient
                chIt.gradientStyle = gradientStyles.Radial
                chIt.gradientRadius = COEFF_PAD * 3
                chIt.backgroundColor = UIColor.brownColor()
                // Add chart item into series
                srs2.Items.addObject(chIt)
            }
            // Add series into chart
            chrt.series.addObject(srs0)
            chrt.series.addObject(srs1)
            chrt.series.addObject(srs2)
        } else if (chrt.chartType == chartTypes.point) {
            lblSampleTitle.text = "Point Chart"
            
            // Define chart's Layout style
            chrt.layoutStyle = layoutStyles.Gradient
            // Define chart's gradient
            chrt.gradientStyle = gradientStyles.Radial
            chrt.gradientRadius = 200
            chrt.gradientColorTop = UIColor(red: 0.1, green:0.8, blue:0.1, alpha:0.4)
            chrt.gradientColorTopExtra = UIColor.clearColor()
            chrt.gradientColorBottom = UIColor.clearColor()
            chrt.gradientColorBottomExtra = UIColor(red: 0.1, green:0.1, blue:0.8, alpha:0.4)
            chrt.gradientDirection = gradientDirections.Horizontal
            
            // Assign chart's Title
            chrt.chartTitle.text = "Prevalence of 3 types of apples across Continents"
            chrt.chartTitle.font = UIFont(name:"Georgia", size:COEFF_PAD * 13)
            chrt.chartTitle.backgroundColor = UIColor(red: 0.1, green:0.1, blue:0.8, alpha:0.1)
            chrt.chartTitle.textColor = UIColor.whiteColor()
            chrt.chartTitle.frame = CGRectMake(COEFF_PAD * 5, COEFF_PAD * 5, COEFF_PAD * 310, COEFF_PAD * 18)
            
            // Add marked region into chart plot area
            let markedRgn = markedRegion()
            markedRgn.yTop = 0.5
            markedRgn.yBottom = 0.3
            markedRgn.Title.text = "Marked Area"
            markedRgn.Title.textColor = UIColor.yellowColor()
            markedRgn.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 8)
            // Adjust marked region location
            markedRgn.titleVerticalLocation = verticalLocations.Top
            markedRgn.titleHorizontalLocation = horizontalLocations.Right
            chrt.markedRegions.addObject(markedRgn)
            
            // Set X axis title
            chrt.xAxisTitleLabel.text = "Continents"
            chrt.xAxisTitleLabel.textColor = UIColor.purpleColor()
            chrt.xAxisTitleLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 12)
            chrt.xAxisTitleHorizontalLocation = horizontalLocations.Right
            // X axis labels
            chrt.xAxisLinesHidden = false
            chrt.xAxisLabelsHidden = false
            chrt.xAxisLabelsFont = UIFont(name:"Georgia", size:COEFF_PAD * 6)
            chrt.xAxisLabelsTextColor = UIColor.yellowColor()
            // Y axis labels
            chrt.yAxisLabelsFont = UIFont(name:"Georgia", size:COEFF_PAD * 6)
            chrt.yAxisLabelsTextColor = UIColor.blueColor()
            
            // Draw Legend Zone
            chrt.legendZone.layer.borderColor = UIColor.yellowColor().CGColor
            chrt.legendZone.layer.borderWidth = COEFF_PAD * 0.5
            chrt.legendsInterpositions = orientations.horizontal
            chrt.legendZoneHorizontalLocation = horizontalLocations.Mid
            chrt.legendZoneVerticalLocation = verticalLocations.Bottom
            
            // Create legend items to add into the legend zone
           var lgd = legend()
            lgd.legendLabel.text = "Rejuvenating Apple"
            lgd.legendLabel.textColor = UIColor.whiteColor()
            lgd.legendLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 7)
            lgd.legendBulletType = legendBulletTypes.Triangle
            lgd.legendBulletWidth = COEFF_PAD * 12
            lgd.legendBulletFillColor = UIColor.whiteColor()
            lgd.legendBulletStrokeLineColor = UIColor.redColor()
            lgd.legendBulletStrokeLineWidth = COEFF_PAD * 2
            lgd.legendMargin = COEFF_PAD * 3
            chrt.legends.addObject(lgd)
            lgd = legend()
            lgd.legendLabel.text = "Apple of Discord"
            lgd.legendLabel.textColor = UIColor.whiteColor()
            lgd.legendLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 7)
            lgd.legendBulletType = legendBulletTypes.Dot
            lgd.legendBulletFillColor = UIColor.redColor()
            lgd.legendBulletWidth = COEFF_PAD * 12
            lgd.legendMargin = COEFF_PAD * 3
            chrt.legends.addObject(lgd)
            lgd = legend()
            lgd.legendLabel.text = "Hesperides Apple"
            lgd.legendLabel.textColor = UIColor.whiteColor()
            lgd.legendLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 7)
            lgd.legendBulletType = legendBulletTypes.Rect
            lgd.legendBulletFillColor = UIColor.blueColor()
            lgd.legendBulletWidth = COEFF_PAD * 12
            lgd.legendMargin = COEFF_PAD * 3
            chrt.legends.addObject(lgd)
            
            // Define chart series
            let srs0 = ChartSeries()
            let srs1 = ChartSeries()
            let srs2 = ChartSeries()
            
            // Fill out chart items of 1st series
            // Define the series' line shape
            srs0.shape.lineWidth = COEFF_PAD * 0.3
            for var i = 0; i < dataItmsSeries1.count; i++ {
                var itm = dataItmsSeries1.objectAtIndex(i) as NSMutableArray
                let chIt = ChartItem()
                chIt.value = CGFloat((itm.objectAtIndex(1) as NSString).doubleValue)
                chIt.Title.text = itm.objectAtIndex(1) as NSString
                chIt.Title.textColor = UIColor.whiteColor()
                chIt.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
                chIt.layer.borderColor = UIColor.whiteColor().CGColor
                chIt.layer.borderWidth = COEFF_PAD * 0.3
                chIt.titleVerticalLocation = verticalLocations.Bottom
                chIt.titleHorizontalLocation = horizontalLocations.Mid
                chIt.pointType = pointTypes.Triangle
                chIt.pointWidth = COEFF_PAD * 9
                chIt.pointFillColor = UIColor.whiteColor()
                chIt.pointStrokeLineWidth = COEFF_PAD * 0.2
                chIt.pointStrokeLineColor = UIColor.redColor()
                chIt.layoutStyle = layoutStyles.Flat
                // X axis label text
                chIt.xAxisLabel.text = itm.objectAtIndex(0) as NSString
                // Add chart item into series
                srs0.Items.addObject(chIt)
            }
            
            // Fill out chart items of 2nd series
            // Define the series' line shape
            srs1.shape.lineWidth = COEFF_PAD * 3
            srs1.shape.strokeColor = UIColor.redColor().CGColor
            // Define line shadow
            srs1.shape.shadowColor = UIColor.blackColor().CGColor
            srs1.shape.shadowOpacity = 0.9
            srs1.shape.shadowOffset = CGSizeMake(COEFF_PAD * 10, COEFF_PAD * 10)
            srs1.shape.shadowRadius = 10
            srs1.shape.shadowPath = srs0.shape.path
            for var i = 0; i < dataItmsSeries0.count; i++ {
                var itm = dataItmsSeries0.objectAtIndex(i) as NSMutableArray
                let chIt = ChartItem()
                chIt.value = CGFloat((itm.objectAtIndex(1) as NSString).doubleValue)
                chIt.Title.text = itm.objectAtIndex(1) as NSString
                chIt.Title.textColor = UIColor.blackColor()
                chIt.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
                chIt.titleVerticalLocation = verticalLocations.Top
                chIt.titleHorizontalLocation = horizontalLocations.Mid
                chIt.pointWidth = 0
                chIt.pointType = pointTypes.Dot
                chIt.pointWidth = COEFF_PAD * 9
                chIt.pointFillColor = UIColor.redColor()
                chIt.layoutStyle = layoutStyles.Flat
                // Add chart item into series
                srs1.Items.addObject(chIt)
            }
            
            // Fill out chart items of 3rd series
            // Define the series' line shape
            srs2.shape.lineWidth = COEFF_PAD * 3
            srs2.shape.strokeColor = UIColor.blueColor().CGColor
            // Define line shadow
            srs2.shape.shadowColor = UIColor.blackColor().CGColor
            srs2.shape.shadowOpacity = 0.9
            srs2.shape.shadowOffset = CGSizeMake(COEFF_PAD * 10, COEFF_PAD * 10)
            srs2.shape.shadowRadius = 10
            srs2.shape.shadowPath = srs0.shape.path
            for var i = 0; i < dataItmsSeries2.count; i++ {
                var itm = dataItmsSeries2.objectAtIndex(i) as NSMutableArray
                let chIt = ChartItem()
                chIt.value = CGFloat((itm.objectAtIndex(1) as NSString).doubleValue) * (swhNegativeSeries.on ? -1 : 1)
                chIt.Title.text = NSString(format: "%.2f", chIt.value)
                chIt.Title.textColor = UIColor.blueColor()
                chIt.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
                chIt.titleVerticalLocation = verticalLocations.Top
                chIt.titleHorizontalLocation = horizontalLocations.Mid
                chIt.pointType = pointTypes.Rect
                chIt.pointWidth = COEFF_PAD * 9
                chIt.pointFillColor = UIColor.blueColor()
                chIt.layoutStyle = layoutStyles.Flat
                // Add chart item into series
                srs2.Items.addObject(chIt)
            }
            // Add series into chart
            chrt.series.addObject(srs0)
            chrt.series.addObject(srs1)
            chrt.series.addObject(srs2)
        } else if (chrt.chartType == chartTypes.cylinder) {
            lblSampleTitle.text = "Cylinder Chart"
            
            // Define chart's Layout style
            chrt.layoutStyle = layoutStyles.Gradient
            chrt.gradientStyle = gradientStyles.Cross
            chrt.gradientColorTop = UIColor(red: 0.2, green:0.1, blue:0.1, alpha:0.1)
            chrt.gradientColorTopExtra = UIColor.clearColor()
            chrt.gradientColorBottom = UIColor(red: 0.2, green:0.1, blue:0.1, alpha:0.1)
            chrt.gradientColorBottomExtra = UIColor(red: 0.2, green:0.05, blue:0.05, alpha:0.1)
            chrt.gradientDirection = gradientDirections.DiagonalLeft
            
            // Also you may define chart's layout as a background image
            //chrt.layoutStyle = layoutStyles.Image
            //chrt.backgroundImageName = "background.png"
            
            // Assign chart's Title
            chrt.chartTitle.text = "Some Apples are Sweet, some tart..."
            chrt.chartTitle.font = UIFont(name:"Georgia", size:COEFF_PAD * 16)
            chrt.chartTitle.backgroundColor = UIColor(red: 0.1, green:0.1, blue:0.8, alpha:0.1)
            chrt.chartTitle.textColor = UIColor.lightTextColor()
            
            // Set X axis title
            chrt.xAxisTitleLabel.text = "Apples"
            chrt.xAxisTitleLabel.textColor = UIColor.orangeColor()
            chrt.xAxisTitleLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 12)
            chrt.xAxisTitleHorizontalLocation = horizontalLocations.Right
            chrt.xAxisTitleLabelRotationAngle = -10
            chrt.xAxisTitleXshift = 11
            chrt.xAxisTitleYshift = 25
            // X axis labels
            chrt.xAxisLinesHidden = false
            chrt.xAxisLabelsHidden = false
            chrt.xAxisLabelsFont = UIFont(name:"Georgia", size:COEFF_PAD * 6)
            chrt.xAxisLabelsTextColor = UIColor.yellowColor()
            // Y axis labels
            chrt.yAxisLabelsFont = UIFont(name:"Georgia", size:COEFF_PAD * 6)
            chrt.yAxisLabelsTextColor = UIColor.blueColor()
            
            // Draw Legend Zone
            chrt.legendZone.layer.borderColor = UIColor.yellowColor().CGColor
            chrt.legendZone.layer.borderWidth = COEFF_PAD * 0.5
            chrt.legendsInterpositions = orientations.vertical
            chrt.legendZoneHorizontalLocation = horizontalLocations.Right
            chrt.legendZoneVerticalLocation = verticalLocations.Top
            chrt.legendZone.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 8)
            chrt.legendZone.Title.text = "Best for "
            chrt.legendZone.Title.textColor = UIColor.yellowColor()
            chrt.legendZone.titleVerticalLocation = verticalLocations.Mid
            chrt.legendZone.titleHorizontalLocation = horizontalLocations.Left
            
            // Create legend items to add into the legend zone
           var lgd = legend()
            lgd.legendLabel.text = "Cooking"
            lgd.legendLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 7)
            lgd.legendBulletType = legendBulletTypes.Image
            lgd.legendBulletImageName = "flwr_red.gif"
            lgd.legendBulletWidth = COEFF_PAD * 12
            chrt.legends.addObject(lgd)
            lgd = legend()
            lgd.legendLabel.text = "Baking"
            lgd.legendLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 7)
            lgd.legendBulletType = legendBulletTypes.Image
            lgd.legendBulletImageName = "flwr_white.gif"
            lgd.legendBulletWidth = COEFF_PAD * 12
            chrt.legends.addObject(lgd)
            lgd = legend()
            lgd.legendLabel.text = "Eating Fresh"
            lgd.legendBulletType = legendBulletTypes.Image
            lgd.legendLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 7)
            lgd.legendBulletImageName = "flwr_blue.gif"
            lgd.legendBulletWidth = COEFF_PAD * 12
            chrt.legends.addObject(lgd)
            
            // Define chart series
            let srs0 = ChartSeries()
            let srs1 = ChartSeries()
            let srs2 = ChartSeries()
            
            // Fill out chart items of 1st series
            for var i = 0; i < dataItmsSeries0.count; i++ {
                var itm = dataItmsSeries0.objectAtIndex(i) as NSMutableArray
                let chIt = ChartItem()
                chIt.value = CGFloat((itm.objectAtIndex(1) as NSString).doubleValue)
                // Item Title text
                chIt.Title.text = itm.objectAtIndex(1) as NSString
                chIt.Title.textColor = UIColor.blackColor()
                chIt.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
                // Item Title location
                chIt.titleVerticalLocation = verticalLocations.Mid
                chIt.titleHorizontalLocation = horizontalLocations.Mid
                // Define item's border
                chIt.layer.borderColor = UIColor.whiteColor().CGColor
                chIt.layer.borderWidth = COEFF_PAD * 0.3
                chIt.layoutStyle = layoutStyles.Flat
                chIt.backgroundColor = UIColor.redColor()
                // X axis label text
                chIt.xAxisLabel.text = itm.objectAtIndex(2) as NSString
                // Add chart item into series
                srs0.Items.addObject(chIt)
            }
            
            // Fill out chart items of 2nd series
            for var i = 0; i < dataItmsSeries1.count; i++ {
                var itm = dataItmsSeries1.objectAtIndex(i) as NSMutableArray
                let chIt = ChartItem()
                chIt.value = CGFloat((itm.objectAtIndex(1) as NSString).doubleValue)
                chIt.Title.text = itm.objectAtIndex(1) as NSString
                chIt.Title.textColor = UIColor.darkGrayColor()
                chIt.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
                chIt.titleHorizontalLocation = horizontalLocations.Mid
                chIt.titleVerticalLocation = verticalLocations.Mid
                chIt.layoutStyle = layoutStyles.Flat
                chIt.backgroundColor = UIColor.whiteColor()
                // Add chart item into series
                srs1.Items.addObject(chIt)
            }
            
            // Fill out chart items of 3rd series
            for var i = 0; i < dataItmsSeries2.count; i++ {
                var itm = dataItmsSeries2.objectAtIndex(i) as NSMutableArray
                let chIt = ChartItem()
                chIt.value = CGFloat((itm.objectAtIndex(1) as NSString).doubleValue) * (swhNegativeSeries.on ? -1 : 1)
                chIt.Title.text = NSString(format: "%.2f", chIt.value)
                chIt.Title.textColor = UIColor.yellowColor()
                chIt.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
                chIt.titleHorizontalLocation = horizontalLocations.Mid
                chIt.titleVerticalLocation = verticalLocations.Mid
                chIt.pointType = pointTypes.Rect
                // Define item's border
                chIt.layer.borderWidth = COEFF_PAD * 0.5
                // Define item's layout style
                chIt.layoutStyle = layoutStyles.Flat
                chIt.backgroundColor = UIColor.blueColor()
                // Add chart item into series
                srs2.Items.addObject(chIt)
            }
            // Add series into chart
            chrt.series.addObject(srs0)
            chrt.series.addObject(srs1)
            chrt.series.addObject(srs2)
            
            // Proportion width of bar/cylinder item (range: 0.0 - 1.0) Default value is 0.5
            chrt.itemBreadth = 0.6
        } else if (chrt.chartType == chartTypes.cylinderStacked) {
            lblSampleTitle.text = "Stacked Cylinder Chart"
            
            // Define chart's Layout style
            chrt.layoutStyle = layoutStyles.Gradient
            chrt.gradientStyle = gradientStyles.Square
            chrt.gradientColorTop = UIColor(red: 0.2, green:0.1, blue:0.1, alpha:0.1)
            chrt.gradientColorTopExtra = UIColor.clearColor()
            chrt.gradientColorBottom = UIColor(red: 0.2, green:0.1, blue:0.1, alpha:0.1)
            chrt.gradientColorBottomExtra = UIColor(red: 0.2, green:0.05, blue:0.05, alpha:0.1)
            chrt.gradientDirection = gradientDirections.Horizontal
            
            // Assign chart's Title
            chrt.chartTitle.text = "Some Apples are Sweet, some tart..."
            chrt.chartTitle.font = UIFont(name:"Georgia", size:COEFF_PAD * 16)
            chrt.chartTitle.backgroundColor = UIColor(red: 0.1, green:0.1, blue:0.8, alpha:0.1)
            chrt.chartTitle.textColor = UIColor.lightTextColor()
            
            // Set X axis title
            chrt.xAxisTitleLabel.text = "Apples"
            chrt.xAxisTitleLabel.textColor = UIColor.orangeColor()
            chrt.xAxisTitleLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 12)
            chrt.xAxisTitleHorizontalLocation = horizontalLocations.Right
            chrt.xAxisTitleLabelRotationAngle = -10
            chrt.xAxisTitleXshift = 11
            chrt.xAxisTitleYshift = 25
            // X axis labels
            chrt.xAxisLinesHidden = false
            chrt.xAxisLabelsHidden = false
            chrt.xAxisLabelsFont = UIFont(name:"Georgia", size:COEFF_PAD * 6)
            chrt.xAxisLabelsTextColor = UIColor.yellowColor()
            // Y axis labels
            chrt.yAxisLabelsFont = UIFont(name:"Georgia", size:COEFF_PAD * 6)
            chrt.yAxisLabelsTextColor = UIColor.blueColor()
            
            // Draw Legend Zone
            chrt.legendZone.layer.borderColor = UIColor.yellowColor().CGColor
            chrt.legendZone.layer.borderWidth = COEFF_PAD * 0.5
            chrt.legendsInterpositions = orientations.vertical
            chrt.legendZoneHorizontalLocation = horizontalLocations.Right
            chrt.legendZoneVerticalLocation = verticalLocations.Top
            chrt.legendZone.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 8)
            chrt.legendZone.Title.text = "Best for "
            chrt.legendZone.Title.textColor = UIColor.yellowColor()
            chrt.legendZone.titleVerticalLocation = verticalLocations.Mid
            chrt.legendZone.titleHorizontalLocation = horizontalLocations.Left
            
            // Create legend items to add into the legend zone
           var lgd = legend()
            lgd.legendLabel.text = "Cooking"
            lgd.legendLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 7)
            lgd.legendBulletType = legendBulletTypes.Image
            lgd.legendBulletImageName = "flwr_red.gif"
            lgd.legendBulletWidth = COEFF_PAD * 12
            chrt.legends.addObject(lgd)
            lgd = legend()
            lgd.legendLabel.text = "Baking"
            lgd.legendLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 7)
            lgd.legendBulletType = legendBulletTypes.Image
            lgd.legendBulletImageName = "flwr_white.gif"
            lgd.legendBulletWidth = COEFF_PAD * 12
            chrt.legends.addObject(lgd)
            lgd = legend()
            lgd.legendLabel.text = "Eating Fresh"
            lgd.legendBulletType = legendBulletTypes.Image
            lgd.legendLabel.font = UIFont(name:"Georgia", size:COEFF_PAD * 7)
            lgd.legendBulletImageName = "flwr_blue.gif"
            lgd.legendBulletWidth = COEFF_PAD * 12
            chrt.legends.addObject(lgd)
            
            // Define chart series
            let srs0 = ChartSeries()
            let srs1 = ChartSeries()
            let srs2 = ChartSeries()
            
            // Fill out chart items of 1st series
            for var i = 0; i < dataItmsSeries0.count; i++ {
                var itm = dataItmsSeries0.objectAtIndex(i) as NSMutableArray
                let chIt = ChartItem()
                chIt.value = CGFloat((itm.objectAtIndex(1) as NSString).doubleValue)
                // Item Title text
                chIt.Title.text = itm.objectAtIndex(1) as NSString
                chIt.Title.textColor = UIColor.blackColor()
                chIt.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
                // Item Title location
                chIt.titleVerticalLocation = verticalLocations.Mid
                chIt.titleHorizontalLocation = horizontalLocations.Mid
                // Define item's border
                chIt.layer.borderColor = UIColor.whiteColor().CGColor
                chIt.layer.borderWidth = COEFF_PAD * 0.3
                chIt.layoutStyle = layoutStyles.Flat
                chIt.backgroundColor = UIColor.redColor()
                // X axis label text
                chIt.xAxisLabel.text = itm.objectAtIndex(2) as NSString
                // Add chart item into series
                srs0.Items.addObject(chIt)
            }
            
            // Fill out chart items of 2nd series
            for var i = 0; i < dataItmsSeries1.count; i++ {
                var itm = dataItmsSeries1.objectAtIndex(i) as NSMutableArray
                let chIt = ChartItem()
                chIt.value = CGFloat((itm.objectAtIndex(1) as NSString).doubleValue)
                chIt.Title.text = itm.objectAtIndex(1) as NSString
                chIt.Title.textColor = UIColor.darkGrayColor()
                chIt.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
                chIt.titleHorizontalLocation = horizontalLocations.Mid
                chIt.titleVerticalLocation = verticalLocations.Mid
                chIt.layoutStyle = layoutStyles.Flat
                chIt.backgroundColor = UIColor.whiteColor()
                // Add chart item into series
                srs1.Items.addObject(chIt)
            }
            
            // Fill out chart items of 3rd series
            for var i = 0; i < dataItmsSeries2.count; i++ {
                var itm = dataItmsSeries2.objectAtIndex(i) as NSMutableArray
                let chIt = ChartItem()
                chIt.value = CGFloat((itm.objectAtIndex(1) as NSString).doubleValue) * (swhNegativeSeries.on ? -1 : 1)
                chIt.Title.text = NSString(format: "%.2f", chIt.value)
                chIt.Title.textColor = UIColor.yellowColor()
                chIt.Title.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
                chIt.titleHorizontalLocation = horizontalLocations.Mid
                chIt.titleVerticalLocation = verticalLocations.Mid
                chIt.pointType = pointTypes.Rect
                // Define item's border
                chIt.layer.borderWidth = COEFF_PAD * 0.5
                // Define item's layout style
                chIt.layoutStyle = layoutStyles.Flat
                chIt.backgroundColor = UIColor.blueColor()
                // Add chart item into series
                srs2.Items.addObject(chIt)
            }
            // Add series into chart
            chrt.series.addObject(srs0)
            chrt.series.addObject(srs1)
            chrt.series.addObject(srs2)
            
            // Proportion width of bar/cylinder item (range: 0.0 - 1.0) Default value is 0.5
            chrt.itemBreadth = 0.9
        }
    }
    
    // Define data source for chart items in 3 Series
    func dataItemsSeries() {

        // Data source for 1st Series
        var itm = NSMutableArray()
        itm.addObject("Anthony")
        itm.addObject("0.40")
        itm.addObject("Ambrosia")
        dataItmsSeries0.addObject(itm)
        itm = NSMutableArray()
        itm.addObject("Mary")
        itm.addObject("0.5")
        itm.addObject("Pink Lady")
        dataItmsSeries0.addObject(itm)
        itm = NSMutableArray()
        itm.addObject("Russ")
        itm.addObject("0.45")
        itm.addObject("McIntosh")
        dataItmsSeries0.addObject(itm)
        itm = NSMutableArray()
        itm.addObject("Maxwell")
        itm.addObject("0.35")
        itm.addObject("Fuji")
        dataItmsSeries0.addObject(itm)
        itm = NSMutableArray()
        itm.addObject("Tom")
        itm.addObject("0.7")
        itm.addObject("Gala")
        dataItmsSeries0.addObject(itm)
        itm = NSMutableArray()
        itm.addObject("Michell")
        itm.addObject("0.60")
        itm.addObject("Golden Delicious")
        dataItmsSeries0.addObject(itm)

        // Data source for 2nd Series
        itm = NSMutableArray()
        itm.addObject("Hyperborea")
        itm.addObject("0.50")
        dataItmsSeries1.addObject(itm)
        itm = NSMutableArray()
        itm.addObject("Atlantis")
        itm.addObject("0.2")
        dataItmsSeries1.addObject(itm)
        itm = NSMutableArray()
        itm.addObject("Arktida")
        itm.addObject("0.65")
        dataItmsSeries1.addObject(itm)
        itm = NSMutableArray()
        itm.addObject("Lemuria")
        itm.addObject("0.45")
        dataItmsSeries1.addObject(itm)
        itm = NSMutableArray()
        itm.addObject("Rutas")
        itm.addObject("0.8")
        dataItmsSeries1.addObject(itm)
        itm = NSMutableArray()
        itm.addObject("Pacifis")
        itm.addObject("0.5")
        dataItmsSeries1.addObject(itm)

        // Data source for 3rd Series
        itm = NSMutableArray()
        itm.addObject("RAW")
        itm.addObject("0.80")
        dataItmsSeries2.addObject(itm)
        itm = NSMutableArray()
        itm.addObject("PIES")
        itm.addObject("0.7")
        dataItmsSeries2.addObject(itm)
        itm = NSMutableArray()
        itm.addObject("BAKING")
        itm.addObject("0.55")
        dataItmsSeries2.addObject(itm)
        itm = NSMutableArray()
        itm.addObject("SAUCE")
        itm.addObject("0.6")
        dataItmsSeries2.addObject(itm)
        itm = NSMutableArray()
        itm.addObject("JUICE")
        itm.addObject("0.9")
        dataItmsSeries2.addObject(itm)
        itm = NSMutableArray()
        itm.addObject("WINE")
        itm.addObject("0.8")
        dataItmsSeries2.addObject(itm)
    }
    
    // Click btnForward button
    func btnForwardClicked(sender: UIButton) {
        // Set next chart type
        //chartTypes(rawValue: 1)
        
        currentChartType += 1
        btnBackward.alpha = 1
        if (currentChartType == 9) {
            currentChartType = 0
            btnBackward.alpha = 0.5
        }
        swhOrientation.enabled = !(currentChartType == 0)
        lblNegative.text = (currentChartType == 0) ? "non-3D Pie" : "Negative Series"
        // Set up UIDataGrid object's layout to show TextView Column Sample
        chrt.chartType = self.defineChartType()
    
        // Define Chart Layout
        self.defineChart()
    
        // Run Chart
        chrt.DataBind()
        btnForward.frame = CGRectMake(chrt.frame.origin.x + chrt.frame.size.width - 5, chrt.frame.origin.y + chrt.frame.size.height / 2, COEFF_PAD * 15, COEFF_PAD * 15)
        btnBackward.frame = CGRectMake(0, chrt.frame.origin.y + chrt.frame.size.height / 2, COEFF_PAD * 15, COEFF_PAD * 15)
    }
    
    // Click btnBackward button
    func btnBackwardClicked(sender: UIButton) {
        if (currentChartType == 0) { return }
        // Set next chart type
        currentChartType -= 1
        if (currentChartType == 0) { btnBackward.alpha = 0.5 }
        swhOrientation.enabled = !(currentChartType == 0)
        lblNegative.text = (currentChartType == 0) ? "non-3D Pie" : "Negative Series"
        // Set up UIDataGrid object's layout to show TextView Column Sample
        chrt.chartType = self.defineChartType()
    
        // Define Chart Layout
        self.defineChart()
    
        // Run Chart
        chrt.DataBind()
        btnForward.frame = CGRectMake(chrt.frame.origin.x + chrt.frame.size.width - 5, chrt.frame.origin.y + chrt.frame.size.height / 2, COEFF_PAD * 15, COEFF_PAD * 15)
        btnBackward.frame = CGRectMake(0, chrt.frame.origin.y + chrt.frame.size.height / 2, COEFF_PAD * 15, COEFF_PAD * 15)
    }
    
    // Change Chart orientation or add/delete Negative Series
    func swhChanged(sender: UISwitch) {
        // Define Chart Layout
        self.defineChart()
    
        // Run Chart
        chrt.DataBind()
        btnForward.frame = CGRectMake(chrt.frame.origin.x + chrt.frame.size.width - 5, chrt.frame.origin.y + chrt.frame.size.height / 2, COEFF_PAD * 15, COEFF_PAD * 15)
        btnBackward.frame = CGRectMake(0, chrt.frame.origin.y + chrt.frame.size.height / 2, COEFF_PAD * 15, COEFF_PAD * 15)
    }
    
    func defineChartType() -> chartTypes {
        switch currentChartType {
        case 0:
            return chartTypes.pie
        case 1:
            return chartTypes.bar
        case 2:
            return chartTypes.barStacked
        case 3:
            return chartTypes.line
        case 4:
            return chartTypes.area
        case 5:
            return chartTypes.areaStacked
        case 6:
            return chartTypes.point
        case 7:
            return chartTypes.cylinder
        case 8:
            return chartTypes.cylinderStacked
        default:
            return chartTypes.pie
        }
    }
    
    // Event-handling procedure that fires when the chart is tapped
    func chartTapped(sender: UIChart) {
        // Show a message
        self.msgbox("The chartTapped event has been implemented.", title: "Chart is Tapped!")
        return
    }
    
    // Show message box
    func msgbox(msg: NSString, title: NSString) {
        let alert = UIAlertView(title: title, message:msg, delegate: self, cancelButtonTitle: "Ok", otherButtonTitles: "", "")
        alert.show()
    }

}
